--[[

-- DX Ticker v1.0

-- by F6FVY

-- April 2014

--]]

local kLcdChars = 20
local kLcdLines = 4
local kMaxHistory = 100 -- History depth
local kLcdFile = "/tmp/lcd.txt"
local kFifofile = "/tmp/fifo"

local bVerbose = false
local kViewList = 0
local kViewDetailed = 1

-- Trim string function

local function trim(s)
  return s:match("^%s*(.-)%s*$")
end

-- Base class

local function clusterData()
  
  local self = { -- Public fields
    historyLine = 0,
    maxLine = 0,
    lcdLine = 0,
    view = kViewList,

    rawData = {}, -- detailed view
    lcdData = {}  -- list view
  }

  -- private

  --

  function self.fixScroll()
    local bScrollBack = (self.lcdLine ~= self.historyLine)
    if #self.lcdData > kMaxHistory then
      table.remove(self.lcdData, 1)
      table.remove(self.rawData, 1)
      if bScrollBack then -- Scrollback in progress
        if self.lcdLine > 1 then
          self.lcdLine = self.lcdLine - 1 -- Display current data
        end
      end
    else
      self.historyLine = self.historyLine + 1
      if (not bScrollBack) then -- No scrollback - always display latest line
        self.lcdLine = self.historyLine
      end
      if self.historyLine > self.maxLine then self.maxLine = self.historyLine end
    end
  end

  function self.printLcdLine(file, line)
    if (line) then
      if (bVerbose) then print(line) end
      file:write(line  .. '\n')
    end
  end

  function self.updateLcd()
    if (bVerbose) then os.execute("clear") end
    local f = assert(io.open(kLcdFile, "w"))
    if (self.lcdLine <= 0) then self.printLcdLine(f, "No data yet...")
    else
      local i
      for i = 1 - kLcdLines, 0 do
        local l = self.lcdLine + i
        if (l >= 1) then
          self.printLcdLine(f, self.lcdData[l])
        else
          self.printLcdLine(f, "")
        end
      end
    end
    assert(f:close())
  end

  return self
end

-- Derived class (DX Spots)

local function spotData()
  local self = clusterData()

  function self.appendData(line)
    local from, freq, dx = line:match('^DX de (.+%d)[%a%d-]+:%s+([%d.]+)%s+([%a%d/]+)')
    if (from == nil) then return false
    else
      self.rawData[#self.rawData + 1] = line

      local nfreq = freq + 0.0
      if (nfreq >= 24000000) then freq = "24g" .. freq:sub(-5)
      elseif (nfreq > 10000000) then freq = "10g" .. freq:sub(-5)
      elseif (nfreq > 5700000) then freq = "5g" .. freq:sub(-5)
      elseif (nfreq > 2300000) then freq = "2g" .. freq:sub(-5)
      elseif (nfreq > 1296000) then freq = "1g" .. freq:sub(-5)
      elseif (nfreq > 432000) then freq = "70c" .. freq:sub(-5)
      elseif (nfreq > 144000) then freq = "2m" .. freq:sub(-5)
      elseif (nfreq < 1000) then freq = "  " .. freq
      elseif (nfreq < 10000) then freq = " " .. freq
      end

      local spot = freq .. " " .. dx .. " "

      if (spot:len() + from:len()) < kLcdChars then
        spot = spot .. string.rep(" ", kLcdChars - (spot:len() + from:len())) .. from
      else
        spot = string.sub(spot .. from, 1, kLcdChars)
      end
      self.lcdData[#self.lcdData + 1] = spot

      self.fixScroll()
      return true
    end
  end

  local printLcdLineBase = self.printLcdLine

  function self.updateLcdDetails()
    if (bVerbose) then os.execute("clear") end

    if (self.lcdLine <= 0) then
      local f = assert(io.open(kLcdFile, "w"))
      printLcdLineBase(f, "No data yet...");
      assert(f:close())      
      return
    end

    local from, freq, dx, comment, time = self.rawData[self.lcdLine]:match('^DX (de .+):(%s+[%d.]+)%s+([%a%d/]+)%s+(.+)%s+(%d%d%d%d)Z')

    comment = trim(comment)
    local comment2 -- Second line content
    if comment:len() > kLcdChars then -- Long line : Cut gracefully into two lines
      local comment20 = comment:sub(1, kLcdChars + 1)
      local comment1
      local i = 0
      while ((comment1 == nil) and (i ~= -comment:len())) do -- Search first space char backwards
        i = i - 1
        comment1 = string.find(comment20, " ", i)
      end

      if (comment1 == nil) then -- No space found ?!
        comment2 = comment:sub(kLcdChars + 1)
        comment = comment:sub(1, kLcdChars)
      else
        comment2 = comment:sub(comment1 + 1, comment:len())
        comment = comment:sub(1, comment1)
      end
    else
      comment2 = ""
    end

    local f = assert(io.open(kLcdFile, "w"))
    printLcdLineBase(f, from .. freq);
    local pdx = "<< " .. dx .. " >>"
    if (pdx:len() < kLcdChars) then pdx = string.rep(" ", math.floor((kLcdChars - pdx:len()) / 2)) .. pdx end -- Center
    printLcdLineBase(f, pdx);
    printLcdLineBase(f, comment)
    printLcdLineBase(f, string.format("%-" .. (kLcdChars - 5) .. "s %s", comment2:sub(1, kLcdChars - 5), time));
    assert(f:close())
  end

  local updateLcdBase = self.updateLcd

  function self.updateLcd()
    if self.view == 0 then updateLcdBase()
    else self.updateLcdDetails()
    end
  end

  return self

end

-- Derived class (WCY)

local function wcyData()
  local self = clusterData()

  function self.appendData(line)
    local h, k, expK, a, r, sfi = line:match('^WCY de .* <(%d%d)> :%s+K=(%d+)%s+expK=(%d+)%s+A=(%d+)%s+R=(%d+)%s+SFI=(%d+).*')
    if (h == nil) then return false
    else
      self.rawData[#self.rawData + 1] = line

      local wcy = h .. " K" .. k .. " A" .. a .. " N" .. r .. " F" .. sfi
      wcy = string.sub(wcy, 1, kLcdChars)
      self.lcdData[#self.lcdData + 1] = wcy

      self.fixScroll()
      return true
    end
  end

  local printLcdLineBase = self.printLcdLine

  function self.updateLcdDetails()
    if (bVerbose) then os.execute("clear") end

    if (self.lcdLine <= 0) then
      local f = assert(io.open(kLcdFile, "w"))
      printLcdLineBase(f, "No data yet...");
      assert(f:close())      
      return
    end

    local l1, l2, l3, l4 = self.rawData[self.lcdLine]:match('^(WCY de .* <%d%d>) :%s+(K=%d+%s+expK=%d+%s+A=%d+)%s+(R=%d+%s+SFI=%d+)%s+(.*)$')

    l1 = trim(l1)
    l2 = trim(l2)
    l3 = trim(l3)
    l4 = trim(l4)

    local f = assert(io.open(kLcdFile, "w"))
    printLcdLineBase(f,l1);
    printLcdLineBase(f,l2);
    printLcdLineBase(f,l3);
    printLcdLineBase(f,l4);
    assert(f:close())
  end

  local updateLcdBase = self.updateLcd

  function self.updateLcd()
    if self.view == 0 then updateLcdBase()
    else self.updateLcdDetails()
    end
  end

  return self

end

-- Derived class (WWV)

local function wwvData()
  local self = clusterData()

  function self.appendData(line)
    local h, sfi, a, k = line:match('^WWV de .* <(%d%d)>:%s+(SFI=%d+),%s+(A=%d+),%s+(K=%d+),.*')
    if (h == nil) then return false
    else
      self.rawData[#self.rawData + 1] = line

      local wwv = h .. " " .. sfi .. " " .. a .. " " .. k
      wwv = string.sub(wwv, 1, kLcdChars)
      self.lcdData[#self.lcdData + 1] = wwv

      self.fixScroll()
      return true
    end
  end

  local printLcdLineBase = self.printLcdLine

  function self.updateLcdDetails()
    if (bVerbose) then os.execute("clear") end

    if (self.lcdLine <= 0) then
      local f = assert(io.open(kLcdFile, "w"))
      printLcdLineBase(f, "No data yet...");
      assert(f:close())      
      return
    end

    local l1, l2, l3, l4 = self.rawData[self.lcdLine]:match('^(WWV de .* <%d%d>):%s+(SFI=%d+,%s+A=%d+,%s+K=%d+),%s+(.*)-(>.*)$')

    l1 = trim(l1)
    l2 = trim(l2)
    l3 = trim(l3)
    l4 = trim(l4)

    local f = assert(io.open(kLcdFile, "w"))
    printLcdLineBase(f,l1);
    printLcdLineBase(f,l2);
    printLcdLineBase(f,l3);
    printLcdLineBase(f,l4);
    assert(f:close())
  end

  local updateLcdBase = self.updateLcd

  function self.updateLcd()
    if self.view == 0 then updateLcdBase()
    else self.updateLcdDetails()
    end
  end

  return self

end

-- Main --

bVerbose = ((arg[1] == "v") or (arg[1] == "-v")) -- Verbose : output to screen as well

local bQuit = false

local dxSpots = spotData()
local wcy = wcyData()
local wwv = wwvData()
local dataSet = {dxSpots, wcy, wwv}
local kSetDxSpots = 1
local kSetWcy = 2
local kSetWWv = 3
local currentSet = 1
local currentData = dataSet[currentSet] -- By default

os.execute("mkfifo " .. kFifofile .." > /dev/null 2>&1") -- Create fifo for keypad
os.execute("echo \"@\" > " .. kFifofile .. " &") -- Fake char in fifo to unblock fifo:read()

local fifo = assert(io.open(kFifofile))

-- Keypad polling function
-- Return true if lcd requires update

local function pollKeypad()
  local c = fifo:read()

  if (c == "E") then -- Toggle list / detailed view
    if (currentData.view == kViewList) then currentData.view = kViewDetailed else currentData.view = kViewList end
    return true
  end

  if (c == "8") then -- Up
    if currentData.lcdLine > 1 then
      currentData.lcdLine = currentData.lcdLine - 1
    else return false
    end
  elseif (c == "2") then -- Down
    if currentData.lcdLine < currentData.maxLine then
      currentData.lcdLine = currentData.lcdLine + 1
    else return false
    end
  elseif (c == "7") then -- Home
    if currentData.lcdLine ~= 1 then
      currentData.lcdLine = 1
    else return false
    end
  elseif (c == "1") then -- End
    if currentData.lcdLine ~= currentData.maxLine then
      currentData.lcdLine = currentData.maxLine
    else return false
    end
  elseif (c == "9") then -- PgUp
    if currentData.lcdLine > 1 then
      currentData.lcdLine = currentData.lcdLine - (kLcdLines - 1)
      if currentData.lcdLine < 1 then currentData.lcdLine = 1 end
    else return false
    end
  elseif (c == "3") then -- PgDown
    if currentData.lcdLine < currentData.maxLine then
      currentData.lcdLine = currentData.lcdLine + (kLcdLines - 1)
      if currentData.lcdLine > currentData.maxLine then
        currentData.lcdLine = currentData.maxLine
      end
    else return false
    end
  elseif (c == "4") then -- Previous data
    currentSet = currentSet - 1
    if currentSet < 1 then currentSet = #dataSet end
    currentData = dataSet[currentSet]
  elseif (c == "6") then -- Next data
    currentSet = currentSet + 1
    if currentSet > #dataSet then currentSet = 1 end
    currentData = dataSet[currentSet]
--  elseif (c == "0") then bQuit = true -- Debug
  else
    return false
  end
  return true
end

-- Banner printing function

local function printBanner(line)
  if (line) then 
    local f = assert(io.open(kLcdFile, "a"))
    f:write(line .. '\n')
    assert(f:close())
    if (bVerbose) then print(line) end
  end
end

--

local kHost = "XXXX" -- DX Cluster IP or host name
local khostCallsign = "XXXX" -- DX Cluster callsign
local kPort = 0 -- DX Cluster port
local kMyCallsign = "xxxx-0" -- My callsign (with SSID or not)

local f -- LCD file
f = assert(io.open(kLcdFile, "w")) -- Empty LCD file
assert(f:close())

printBanner("DX Ticker 1.0\n\n(C) F6FVY April 2014")
os.execute("sleep 5")

-- Main loop

while(1) do
  local socket = require("socket")
  local client = socket.tcp()

  client:settimeout(10)

  local connected = false
  while (connected == false) do
    f = assert(io.open(kLcdFile, "w")) -- Empty LCD file
    assert(f:close())

    printBanner("Connecting " .. khostCallsign .. "...")
    os.execute("sleep 1")

    connected, err = client:connect(kHost, kPort)
  
    if (connected == nil) then
      connected = false
      printBanner("Connect error.")
      printBanner(err)
      printBanner("Retry in 5 sec...")
      os.execute("sleep 5")
    end
  end

  printBanner("Connected...")
  os.execute("sleep 2")
  client:send(kMyCallsign .. "\n") -- Send callsign
  os.execute("sleep 1")
  client:send("unset/here" .. "\n") -- Set no here
  printBanner("Waiting data...")

-- cluster parsing loop

  local kTimeOutClient = 0.04 -- In Sec.  Manually adjusted
  client:settimeout(kTimeOutClient)

  local line = ""
  local timeOutCnx = 0;

  while (bQuit == false) do
    local bUpdateLcd = false
    local s, status, partial = client:receive(1) -- Wait for one char or timeout or close

    if (s) then -- One char received
      timeOutCnx = 0 -- Reset timeout
      if s == "\n" then -- Process complete line
        if dxSpots.appendData(line) then
          if (currentData == dxSpots) then bUpdateLcd = true end
        elseif wcy.appendData(line) then
          if (currentData == wcy) then bUpdateLcd = true end
        elseif wwv.appendData(line) then
          if (currentData == wwv) then bUpdateLcd = true end
        else
          bUpdateLcd = pollKeypad()
        end

        line = ""
      else -- not CRLF
        if (s:byte(1) >= 32) then line = line .. s end -- Append s to the current line
      end
    else
      if status == "timeout" then
        timeOutCnx = timeOutCnx + kTimeOutClient
        if (timeOutCnx > (60 * 5)) then -- No activity since 5 mins : Try to reconnect
          break
        else
          bUpdateLcd = pollKeypad()
        end
      else -- Any error, reconnect
        break
      end
    end

    if bUpdateLcd then currentData.updateLcd() end -- LCD file update
 
  end

  client:close()
  f = assert(io.open(kLcdFile, "w")) -- Empty LCD file
  assert(f:close())

  printBanner("Reconnecting in 5 sec.")
  os.execute("sleep 5")

end
